local CVMFLoader	= OOP_Derive( CUtility, CThink, CVMFSettings );

// keys to fix the angle and position on.
local FixupVectors = {
	"origin",
	"springaxis",
	"attachpoint",
	"hingeaxis",
	"point0",
	"point1",
	"point2",
	"position2",
	"slideaxis",
};
local FixupAngles = {
	"angles",
	"movedir",
	"pushdir",
};

// constructor
function CVMFLoader:__ctor( )
	// delay load buffer.
	self.DelayLoadBuffer = {};
	
	// cache
	self.Cache = {};
	
	// spawn counter
	self.Count = 0;
	
end

// think
function CVMFLoader:OnThink( )
	// load the vmfs entity by entity.
	local i;
	for i = table.getn( self.DelayLoadBuffer ), 1, -1 do
		// fetch the vmf
		local active = self.DelayLoadBuffer[i];
	
		// spawned everything?
		if( active.index <= active.vmf:EntityCount() - 1 ) then
			// spawn it.
			local entity_table = self:CreateEntity(
				active.vmf,
				active.index,
				active.absolute,
				active.rotation, active.translation,
				active.surface_normal, active.surface_angle,
				active.position
			);
			if(entity_table) then -- Got no valid entity => Don't pass the following steps
				// add
				table.insert( active.list, entity_table );
								
				// is it our cleanup entity?
				if( entity_table.is_cleanup ) then
					table.insert( active.cleanups, entity_table.entity );
				
				end
			end
			// increment index
			active.index = active.index + 1;
		
		// run activation.
		else
			// create the undo.
			undo.Create( "VMF" );
			undo.SetPlayer( active.pl );
			
			// do final spawn.
			local entity_table;
			for _, entity_table in pairs( active.list ) do
				// spawn it.
				if( entity_table.entity:IsValid() ) then
					// spawn.
					entity_table.entity:Spawn();
				
				end
			
			end
			
			// do final activation.
			local entity_table;
			for _, entity_table in pairs( active.list ) do
				// spawn it.
				if( entity_table.entity:IsValid() ) then
					// activate it.
					self:ActivateEntity(
						entity_table,
						active.cleanups,
						active.weld,
						active.freeze,
						active.unfreeze,
						active.surface_entity,
						active.pl
					);
				
				end
			
			end
			
			// finish the undo.
			undo.Finish( "VMF ( " .. active.filename .. " ) " );
			
			// remove from the buffer
			table.remove( self.DelayLoadBuffer, i );
	
		end
		
	end
	
end

// load vmf.
function CVMFLoader:Load( pl, filename )
	// TODO: Check flags
	
	// our vmf.
	local vmf;
	
	// read the file.
	local filedata = file.Read( "../" .. filename );
	if( filedata ) then
		// create a new vmf
		vmf = VMFSuite.VMF();
		
		// parse it
		vmf:Parse( filedata );

		// cache it?
		if( VMFSuite.Config['cache'] ) then
			self.Cache[ filename ] = vmf;
			
		end
		
	end


	// no vmf?
	if( !vmf ) then return; end
	
	// get the global entity.
	local global = vmf:FindByClass( "vmf_global" );
	
	// delay loading?
	local delay = VMFSuite.Config['force_delay_spawn'];
	if( global ) then
		// fetch the delay key.
		local _, v = global:FindKey( "delay" );
		v = util.tobool( v or false );
		 
		// this vmf uses delay spawning?
		if( !delay ) then
			delay = v;
			
		end
		
	end
	
	// absolute?
	local absolute = false;
	if( global ) then
		// fetch the absolute key
		local _, v = global:FindKey( "delay" );
		absolute = util.tobool( v or false );

	end
	
	
	// create the active vmf entry.
	local active = {
		// my vmf!
		vmf = vmf,
		filename = filename,
		absolute = absolute,
		delayed = delay,
	};
	pl.ActiveVMF = active;
	
	// absolute?
	if( absolute ) then
		// load the active vmf.
		self:LoadActiveVMF( pl, { HitPos = Vector( 0, 0, 0 ), Entity = GetWorldEntity() } );
	
	// not absolute.
	else
		// dispatch the preview.
		timer.Simple(
			// delay
			0.4,
			
			// function
			VMFSuite.PreviewSystem.DispatchPreview,
			
			// args
			VMFSuite.PreviewSystem, pl, vmf, filename
			
		);
		
		// pull out the spawn tool.
		CC_GMOD_Tool(
			// player.
			pl,
			
			// tool command.
			"gmod_tool",
			
			// tool to draw.
			{
				"spawn_tool"
			}
			
		);
	
	end
	
end

// load active vmf
function CVMFLoader:LoadActiveVMF( pl, tr )
	// fetch our active vmf
	local active = pl.ActiveVMF;
	if( !active || !active.vmf ) then return; end
	
	// increment spawn counter.
	self.Count = self.Count + 1;
	
	// rename our vmf
	active.vmf:Rename( self.Count );
	
	// get the global entity.
	local global = active.vmf:FindByClass( "vmf_global" );
	
	// weld?
	local weld = util.tobool( pl:GetInfo( "vmf_weld_surface" ) );
	if( global ) then
		// fetch the weld key
		local _, v = global:FindKey( "weld" );
		v = util.tobool( v or false );
		
		// vmf overrides convar if convar is false.
		if( !weld ) then
			weld = v;
			
		end
		
	end
	
	// freeze and unfreeze
	local freeze = util.tobool( pl:GetInfo( "vmf_spawn_frozen" ) );
	local unfreeze = util.tobool( pl:GetInfo( "vmf_spawn_unfrozen" ) );
	
	
	// rotation and translation
	local translation = Vector(
		tonumber( pl:GetInfo( "vmf_ofs_x" ) ),
		tonumber( pl:GetInfo( "vmf_ofs_y" ) ),
		tonumber( pl:GetInfo( "vmf_ofs_z" ) )
	);
	local rotation = Angle(
		tonumber( pl:GetInfo( "vmf_rotation_p" ) ),
		tonumber( pl:GetInfo( "vmf_rotation_y" ) ),
		tonumber( pl:GetInfo( "vmf_rotation_r" ) )
	);
	
	// surface align?
	local surface_normal, surface_angle;
	if( !active.absolute && util.tobool( pl:GetInfo( "vmf_align_surface" ) ) ) then
		// normal
		surface_normal = tr.HitNormal;
		
		// angle
		surface_angle = tr.HitNormal:Angle();
		surface_angle.pitch = surface_angle.pitch + 90;
	
	end
	
	// delay-spawn or insta-spawn?
	if( active.delayed ) then
		// make delay buffer table.
		local entry = {
			pl = pl,
			index = 0,
			list = {},
			cleanups = {},
			surface_normal = surface_normal,
			surface_angle = surface_angle,
			rotation = rotation,
			translation = translation,
			weld = weld,
			freeze = freeze,
			unfreeze = unfreeze,
			position = tr.HitPos,
			surface_entity = tr.Entity,
		};
		table.Merge( entry, active );
		
		// add to the send buffer
		table.insert( self.DelayLoadBuffer, entry );
		
		// erase the preview.
		VMFSuite.PreviewSystem:Clear( pl );
		
		// wipe our active vmf.
		pl.ActiveVMF = nil;
		
	
		//
		return;
	
	end
	
	// cleanup entity.
	local cleanup_entities = {};
	
	// normal entities
	local entities = {};
	
	
	// spawn this vmf.
	local i;
	for i = 0, active.vmf:EntityCount() - 1 do
		// spawn it.
		local entity_table = self:CreateEntity(
			active.vmf,
			i,
			active.absolute,
			rotation, translation,
			surface_normal, surface_angle,
			tr.HitPos
		);
		
		// add
		table.insert( entities, entity_table );
						
		// is it our cleanup entity?
		if( entity_table.is_cleanup ) then
			table.insert( cleanup_entities, entity_table.entity );
		
		end
	
	end
	
	// create the undo.
	undo.Create( "VMF" );
	undo.SetPlayer( pl );
	
	// do final spawn.
	local entity_table;
	for _, entity_table in pairs( entities ) do
		// spawn it.
		if( entity_table.entity:IsValid() ) then
			// spawn.
			entity_table.entity:Spawn();
		
		end
	
	end
	
	// do final activation.
	local entity_table;
	for _, entity_table in pairs( entities ) do
		// spawn it.
		if( entity_table.entity:IsValid() ) then
			// activate it.
			self:ActivateEntity(
				entity_table,
				cleanup_entities,
				weld,
				freeze,
				unfreeze,
				tr.Entity,
				pl
			);
		
		end
	
	end
	
	// finish the undo.
	undo.Finish( "VMF ( " .. active.filename .. " ) " );
	
	// multispawn wipe?
	if( !VMFSuite.Config['multispawn'] ) then
		// erase the preview.
		VMFSuite.PreviewSystem:Clear( pl );
		
		// wipe our active vmf.
		pl.ActiveVMF = nil;
	
	end

end


// create entity from vmf.
function CVMFLoader:CreateEntity( vmf, index, absolute, rotation, translation, surface_normal, surface_angle, pos )
	// fetch the entity from the vmf.
	local ent = vmf:Entity( index );
	
	// create entity
	local entity = ents.Create( ent:Class() );
	if( !entity:IsValid() ) then return; end
	
	// TODO: Add entity handler code.
	//
	
	// setup an entity table.
	local entity_table = {
		material = nil,
		frozen = false,
		parent = nil,
		entity = entity,
		is_cleanup = ent:Class() == "vmf_cleanup",
		color_info = {
			color = Color( 255, 255, 255, 255 ),
			rendermode = RENDERMODE_NORMAL,
			renderfx = 0,
		},
		
	};
	
	// no angles?
	local has_angles = ent:FindKey( "angles" );
	if( !has_angles ) then
		ent:AddKey( "angles", "0 0 0" );
	
	end
	
	// apply keyvalues.
	local j;
	for j = 0, ent:KeyCount() - 1 do
		// fetch keyvalue.
		local key, value = ent:Key( j );
		
		// do some special handling.
		if( key == "model" ) then
			util.PrecacheModel( value );
			entity:SetModel( value );
			
		elseif( key == "message" ) then
			util.PrecacheSound( value );

		elseif( key == "_spawnfrozen" ) then
			entity_table.frozen = true;
			
		elseif( key == "_material" || key == "materialoverride" ) then
			if(value == "gmod/rendertarget") then value = "models/rendertarget" end; -- GM9 backwarts compatibility
			if(value == "gmod/shiny") then value = "models/shiny" end; -- GM9 backwarts compatibility
			entity_table.material = value;
			
		elseif(key == "texture") then
			if(value == "materials/sprites/physgbeam.vmt") then value = "materials/sprites/physbeam.vmt" end; -- GM9 backwarts compatibility
			
		elseif( key == "parentname" ) then
			entity_table.parent = value;
			
		elseif( key == "rendercolor" ) then
			local color = self:StringToColor( value );
			entity_table.color_info.color.r = color.r;
			entity_table.color_info.color.g = color.g;
			entity_table.color_info.color.b = color.b;
			
		elseif( key == "renderamt" ) then
			entity_table.color_info.color.a = tonumber( value );
			
		elseif( key == "rendermode" ) then
			entity_table.color_info.rendermode = tonumber( value );
			
		elseif( key == "renderfx" ) then
			entity_table.color_info.renderfx = tonumber( value );
			
		end
		
	
		// fix origins.
		if( !absolute && table.HasValue( FixupVectors, key ) ) then
			// convert the value to a vector.
			local origin = self:StringToVector( value );
			
			// fix it.
			origin = self:FixupVector( pos, rotation, origin, translation, surface_normal, surface_angle );
			
			// apply
			value = tostring( origin );
			
		// fix angles.
		elseif( !absolute && table.HasValue( FixupAngles, key ) ) then
			// convert the value to a vector.
			local angle = self:StringToVector( value );
			
			// fix it.
			angle = self:FixupAngle( angle, rotation, surface_normal, surface_angle );
			
			// apply
			value = tostring( angle );

		end
	
		// apply the keyvalue.
		entity:SetKeyValue( key, value );

	end
	
	// apply connections.
	local j;
	for j = 0, ent:ConnectionCount() - 1 do
		// fetch connection
		local connection = ent:Connection( j );
		
		// fire it.
		entity:Fire( "AddOutput", connection, 0 );
	
	end
	
	// return the entity table
	return entity_table;

end

// activate entity
function CVMFLoader:ActivateEntity( entity_table, cleanup_list, weld, freeze, unfreeze, surface_entity, pl )
	// fetch entity
	local ent = entity_table.entity;
	
	// activate it.
	ent:Activate();
	
	// is it a logic_collision_pair?
	if( ent:GetClass() == "logic_collision_pair" ) then
		ent:Input( "DisableCollisions", nil, nil, nil );
		
	end
	
	// parent.
	if( entity_table.parent ) then
		ent:Fire( "SetParent", entity_table.parent, 0 );
	
	end
	
	// add this entity to the cleanup if it exists?
	local cleanup_entity;
	for _, cleanup_entity in pairs( cleanup_list ) do
		if( cleanup_entity:IsValid() ) then
			cleanup_entity:AddToCleanup( ent );
		
		end
	
	end

	// apply materials.
	if( entity_table.material ) then
		// set material
		ent:SetMaterial( entity_table.material );
	
		// store
		duplicator.StoreEntityModifier( ent, "material", { MaterialOverride = entity_table.material } );
		
	end
	
	// store color modifiers..
	if( entity_table.color_info ) then
		// construct the data.
		local data = {
			Color = entity_table.color_info.color,
			RenderMode = entity_table.color_info.rendermode,
			RenderFX = entity_table.color_info.renderfx,
			
		};
		
		// store
		duplicator.StoreEntityModifier( ent, "colour", data );
		
	end
	
	// TODO: Add bone modifiers ( inflator and positions )
	//
	
	// TODO: Add entity handler code.
	//
	
	// freeze
	if( ( entity_table.frozen && !unfreeze ) || freeze ) then
		// fetch physics object
		local phys = ent:GetPhysicsObject();
		if( phys:IsValid() ) then
			// freeze it.
			phys:EnableMotion( false );
		
		end
	
	// unfreeze
	elseif( unfreeze ) then
		// fetch physics object
		local phys = ent:GetPhysicsObject();
		if( phys:IsValid() ) then
			// freeze it.
			phys:EnableMotion( true );
		
		end
	
	end

	
	// add to the undo and cleanup.
	undo.AddEntity( ent );
	cleanup.Add( pl, "VMF", ent );
	pl:AddCount( "VMF", ent );
	
	// weld.
	if( weld ) then
		// check movetype.
		if( ent:GetMoveType() == MOVETYPE_VPHYSICS ) then
			// weld.!
			local const = constraint.Weld( ent, surface_entity, 0, 0, 0, true, true );
			
			// weld worked?
			if( const ) then
				ent:DeleteOnRemove( const );
				undo.AddEntity( const );
			
			end
			
		end
		
	end
	
end


// create vmf loading class
VMFSuite.LoadingSystem	= CVMFLoader:create();

